﻿//--------------------------------------------------------------------------------------------
// File : asdMesh.h
// Desc : Asura Mesh Module.
// Copyright(c) Project Asura. All right reserved.
//--------------------------------------------------------------------------------------------

#ifndef __ASDX_MESH_H__
#define __ASDX_MESH_H__

//--------------------------------------------------------------------------------------------
// Includes
//--------------------------------------------------------------------------------------------
#include <d3d11.h>
#include <asdxMath.h>
#include <asdxResMesh.h>
#include <asdxTexture.h>


namespace asdx {

//////////////////////////////////////////////////////////////////////////////////////////////
//  Mesh class
//////////////////////////////////////////////////////////////////////////////////////////////
class Mesh
{
    //========================================================================================
    // list of friend classes and methods.
    //========================================================================================
    /* NOTHING */

public:
    //////////////////////////////////////////////////////////////////////////////////////////
    // MaterialBuffer structure
    //////////////////////////////////////////////////////////////////////////////////////////
    struct MaterialBuffer
    {
        asdx::Vector3    Diffuse;        //!< 拡散反射成分です.
        f32              Alpha;          //!< 透過度です.
        asdx::Vector3    Specular;       //!< 鏡面反射成分です.
        f32              Power;          //!< 鏡面反射強度です.

        MaterialBuffer()
        { /* DO_NOTHING */ }
    };

    //////////////////////////////////////////////////////////////////////////////////////////
    // TextureMap structure
    //////////////////////////////////////////////////////////////////////////////////////////
    struct TextureMap
    {
        asdx::Texture2D   Diffuse;        //!< ディフューズマップです.
        asdx::Texture2D   Specular;       //!< スペキュラーマップです.
        asdx::Texture2D   Normal;         //!< 法線マップです.

        TextureMap()
        : Diffuse ()
        , Specular()
        , Normal  ()
        { /* DO_NOTHING */ }
    };

    //////////////////////////////////////////////////////////////////////////////////////////
    // Material structure
    //////////////////////////////////////////////////////////////////////////////////////////
    struct Material
    {
        MaterialBuffer  Buffer;         //!< マテリアルバッファです.
        TextureMap      TextureMap;     //!< テクスチャマップです.

        Material()
        : Buffer    ()
        , TextureMap()
        { /* DO_NOTHING */ }
    };

    //////////////////////////////////////////////////////////////////////////////////////////
    // Subset structure
    //////////////////////////////////////////////////////////////////////////////////////////
    struct Subset
    {
        u32      IndexOffset;    //!< インデックスオフセットです.
        u32      IndexCount;     //!< インデックスカウントです.
        u32      MaterialID;     //!< マテリアル番号です.
    };

    //========================================================================================
    // public variables.
    //========================================================================================
    /* NOTHING */

    //========================================================================================
    // public methods.
    //========================================================================================

    //----------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------
    Mesh();

    //----------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //----------------------------------------------------------------------------------------
    virtual ~Mesh();

    //----------------------------------------------------------------------------------------
    //! @brief      初期化処理です.
    //----------------------------------------------------------------------------------------
    bool Init( 
        ID3D11Device*        pDevice,
        const asdx::ResMesh& mesh,
        const void*          pShaderBytecode,
        const u32            byteCodeLength,
        const char*          resFolderPath = "../res/",
        const char*          dummyFolderPath = "../res/"
    );

    //----------------------------------------------------------------------------------------
    //! @brief      描画処理です.
    //----------------------------------------------------------------------------------------
    void Draw( ID3D11DeviceContext* pDeviceContext );

    //----------------------------------------------------------------------------------------
    //! @brief      終了処理です.
    //----------------------------------------------------------------------------------------
    void Term();

protected:
    //////////////////////////////////////////////////////////////////////////////////////////
    //  TEXTURE_SLOT enum
    //////////////////////////////////////////////////////////////////////////////////////////
    enum TEXTURE_SLOT
    {
        TEXTURE_SLOT_DIFFUSE  = 0,  //!< ディフューズマップのスロット番号です.
        TEXTURE_SLOT_SPECULAR,      //!< スペキュラーマップのスロット番号です.
        TEXTURE_SLOT_NORMAL,        //!< 法線マップのスロット番号です.
        MAX_TEXTURE_SLOT            //!< テクスチャスロット数です.
    };

    //========================================================================================
    // protected variables.
    //========================================================================================
    ID3D11Buffer*       m_pVB;              //!< 頂点バッファです.
    ID3D11Buffer*       m_pIB;              //!< インデックスバッファです.
    ID3D11Buffer*       m_pMB;              //!< マテリアルバッファです.
    ID3D11InputLayout*  m_pIL;              //!< 入力レイアウトです.
    ID3D11SamplerState* m_pDiffuseSmp;      //!< ディフューズマップのサンプラーステートです.
    ID3D11SamplerState* m_pSpecularSmp;     //!< スペキュラーマップのサンプラーステートです.
    ID3D11SamplerState* m_pNormalSmp;       //!< 法線マップのサンプラーステートです.
    u32                 m_Stride;           //!< ストライドです.
    u32                 m_Offset;           //!< オフセットです.
    u32                 m_SubsetCount;      //!< サブセット数です.
    u32                 m_MaterialCount;    //!< マテリアル数です.
    Mesh::Subset*       m_pSubset;          //!< サブセットです.
    Mesh::Material*     m_pMaterial;        //!< マテリアルです.

    //========================================================================================
    // protected methods
    //========================================================================================

    //----------------------------------------------------------------------------------------
    //! @brief      入力レイアウト生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     shaderByteCode      シェーダのバイトコードです.
    //! @param [in]     byteCodeLength      バイトコードの長さです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateIL( ID3D11Device* pDevice, const void* shaderByteCode, const u32 byteCodeLength );

    //----------------------------------------------------------------------------------------
    //! @brief      頂点バッファ生成時の処理です.
    //!
    //! @parma [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateVB( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      インデックスバッファ生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateIB( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      マテリアルバッファ生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateMB( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      サンプラーステート生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateSmp( ID3D11Device* pDevice );

    //----------------------------------------------------------------------------------------
    //! @brief      サブセット生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateSubset( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      マテリアル生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @param [in]     resPath             テクスチャファイルが格納されているリソースフォルダパスです.
    //! @param [in]     dummyPath           ダミーテクスチャが格納されているフォルダパスです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    virtual bool    OnCreateMaterial( ID3D11Device* pDevice, const asdx::ResMesh& mesh, const char* resPath, const char* dummyPath );

    //----------------------------------------------------------------------------------------
    //! @brief      描画開始時の処理です.
    //!
    //! @param [in]     pDeviceContext      デバイスコンテキストです.
    //----------------------------------------------------------------------------------------
    virtual void    OnDrawBegin( ID3D11DeviceContext* pDeviceContext );

    //----------------------------------------------------------------------------------------
    //! @brief      サブセット描画時の処理です.
    //!
    //! @param [in]     pDeviceContext      デバイスコンテキストです.
    //! @param [in]     index               サブセットインデックスです.
    //----------------------------------------------------------------------------------------
    virtual void    OnDrawSubset( ID3D11DeviceContext* pDeviceContext, const u32 index );

    //----------------------------------------------------------------------------------------
    //! @brief      描画終了時の処理です.
    //----------------------------------------------------------------------------------------
    virtual void    OnDrawEnd( ID3D11DeviceContext* pDeviceContext );

    //----------------------------------------------------------------------------------------
    //! @brief      マテリアル解放時の処理です.
    //----------------------------------------------------------------------------------------
    virtual void    OnTermMaterial();

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトの入力レイアウト生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     shaderByteCode      シェーダのバイトコードです.
    //! @param [in]     byteCodeLength      バイトコードの長さです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    bool    DefOnCreateIL( ID3D11Device* pDevice, const void* shaderByteCode, const u32 byteCodeLength );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトの頂点バッファ生成時の処理です.
    //!
    //! @parma [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    bool    DefOnCreateVB( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのインデックスバッファ生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
     bool    DefOnCreateIB( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのマテリアルバッファ生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    bool    DefOnCreateMB( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのサンプラーステート生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    bool    DefOnCreateSmp( ID3D11Device* pDevice );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのサブセット生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    bool    DefOnCreateSubset( ID3D11Device* pDevice, const asdx::ResMesh& mesh );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのマテリアル生成時の処理です.
    //!
    //! @param [in]     pDevice             デバイスです.
    //! @param [in]     mesh                メッシュです.
    //! @param [in]     resPath             テクスチャファイルが格納されているリソースフォルダパスです.
    //! @param [in]     dummyPath           ダミーテクスチャが格納されているフォルダパスです.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //----------------------------------------------------------------------------------------
    bool    DefOnCreateMaterial( ID3D11Device* pDevice, const asdx::ResMesh& mesh, const char* resPath, const char* dummyPath );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトの描画開始時の処理です.
    //!
    //! @param [in]     pDeviceContext      デバイスコンテキストです.
    //----------------------------------------------------------------------------------------
    void    DefOnDrawBegin( ID3D11DeviceContext* pDeviceContext );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのサブセット描画時の処理です.
    //!
    //! @param [in]     pDeviceContext      デバイスコンテキストです.
    //! @param [in]     index               サブセットインデックスです.
    //----------------------------------------------------------------------------------------
    void    DefOnDrawSubset( ID3D11DeviceContext* pDeviceContext, const u32 index );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトの描画終了時の処理です.
    //----------------------------------------------------------------------------------------
    void    DefOnDrawEnd( ID3D11DeviceContext* pDeviceContext );

    //----------------------------------------------------------------------------------------
    //! @brief      デフォルトのマテリアル解放時の処理です.
    //----------------------------------------------------------------------------------------
    void    DefOnTermMaterial();

private:
    //========================================================================================
    // private variables.
    //========================================================================================
    /* NOTHING */

    //========================================================================================
    // private methods.
    //========================================================================================
    Mesh            ( const Mesh& value );      // アクセス禁止.
    void operator = ( const Mesh& value );      // アクセス禁止.

};


} // namespace asdx

#endif//__ASDX_MESH_H__
