﻿//----------------------------------------------------------------------------------------------
// File : asdxFont.h
// Desc : Font Module.
// Copyright(c) Project Asura. All right reserved.
//----------------------------------------------------------------------------------------------

#ifndef __ASDX_FONT_H__
#define __ASDX_FONT_H__

//----------------------------------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------------------------------
#include <d3d11.h>
#include <asdxMath.h>


namespace asdx {

////////////////////////////////////////////////////////////////////////////////////////////////
// Font class
////////////////////////////////////////////////////////////////////////////////////////////////
class Font
{
    //==========================================================================================
    // list of friend classes and methods.
    //==========================================================================================
    /* NOTHING */

private:
    //==========================================================================================
    // private variables.
    //==========================================================================================
    /* NOTHING */

    //==========================================================================================
    // private methods.
    //==========================================================================================
    /* NOTHING */

protected:
    ////////////////////////////////////////////////////////////////////////////////////////////
    // Vertex structure
    ////////////////////////////////////////////////////////////////////////////////////////////
    struct Vertex
    {
        asdx::Vector3    Position;       //!< 位置座標です.
        asdx::Vector4    Color;          //!< 頂点カラーです.
        asdx::Vector2    TexCoord;       //!< テクスチャ座標です.
    };

    //==========================================================================================
    // protected variables.
    //==========================================================================================
    static const size_t                     InputElementCount = 3;              //!< 入力要素数です.
    static const D3D11_INPUT_ELEMENT_DESC   InputElements[ InputElementCount ]; //!< 入力要素です.
    static const size_t NUM_SPRITES           = 2048;   //!< 最大スプライト数です.
    static const size_t NUM_VERTEX_PER_SPRITE = 4;      //!< 1スプライトあたりの頂点数です.
    static const size_t NUM_INDEX_PER_SPRITE  = 6;      //!< 1スプライトあたりのインデックス数です.

    ID3D11VertexShader*         m_pVS;          //!< 頂点シェーダです.
    ID3D11PixelShader*          m_pPS;          //!< ピクセルシェーダです.
    ID3D11InputLayout*          m_pIL;          //!< 入力レイアウトです.
    ID3D11Buffer*               m_pVB;          //!< 頂点バッファです.
    ID3D11Buffer*               m_pIB;          //!< インデックスバッファです.
    ID3D11Buffer*               m_pCB;          //!< 定数バッファです.
    ID3D11Texture2D*            m_pTexture;     //!< テクスチャです.
    ID3D11ShaderResourceView*   m_pSRV;         //!< シェーダリソースビューです.
    ID3D11SamplerState*         m_pSmp;         //!< サンプラーステートです.
    ID3D11BlendState*           m_pBS;          //!< ブレンドステートです.
    ID3D11DepthStencilState*    m_pDSS;         //!< 深度ステンシルステートです.
    ID3D11RasterizerState*      m_pRS;          //!< ラスタライザーステートです.

    u32             m_SpriteCount;              //!< スプライト数です.
    u32             m_FontWidth;                //!< フォントの横幅です.
    u32             m_FontHeight;               //!< フォントの縦幅です.
    char            m_FontName[ 32 ];           //!< フォント名です.
    f32             m_OffsetU;                  //!< U方向のオフセットです.
    f32             m_OffsetV;                  //!< V方向のオフセットです.
    asdx::Vector2   m_ScreenSize;               //!< スクリーンサイズです.
    asdx::Vector4   m_Color;                    //!< テキストカラーです.
    asdx::Matrix    m_Transform;                //!< 変換行列です.
    char            m_Buffer[ NUM_SPRITES ];    //!< 文字列バッファです.
    Vertex          m_Vertices[ NUM_SPRITES * NUM_VERTEX_PER_SPRITE ];  //!< 頂点データです.

    //==========================================================================================
    // protected methods.
    //==========================================================================================
    /* NOTHING */

public:
    //==========================================================================================
    // public variables.
    //==========================================================================================
    /* NOTHING */

    //==========================================================================================
    // public methods.
    //==========================================================================================

    //------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //------------------------------------------------------------------------------------------
    Font();

    //------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //------------------------------------------------------------------------------------------
    virtual ~Font();

    //------------------------------------------------------------------------------------------
    //! @brief      初期化処理です.
    //!
    //! @param [in]     pDevice         デバイスです.
    //! @param [in]     filename        フォントバイナリファイル名です.
    //! @param [in]     screenWidth     スクリーンの横幅です.
    //! @param [in]     screenHeight    スクリーンの縦幅です.
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //------------------------------------------------------------------------------------------
    bool Init( ID3D11Device* pDevice, const char* filename, f32 screenWidth, f32 screenHeight );

    //------------------------------------------------------------------------------------------
    //! @brief      終了処理です.
    //------------------------------------------------------------------------------------------
    void Term();

    //------------------------------------------------------------------------------------------
    //! @brief      スクリーンサイズを設定します.
    //!
    //! @param [in]     width       スクリーンの横幅.
    //! @param [in]     height      スクリーンの縦幅.
    //------------------------------------------------------------------------------------------
    void SetScreenSize( f32 width, f32 height );

    //------------------------------------------------------------------------------------------
    //! @brief      スクリーンサイズを設定します.
    //!
    //! @param [in]     size        設定するサイズ.
    //------------------------------------------------------------------------------------------
    void SetScreenSize( const asdx::Vector2& size );

    //------------------------------------------------------------------------------------------
    //! @brief      テキストカラーを設定します.
    //!
    //! @param [in]     r       R成分です.
    //! @param [in]     g       G成分です.
    //! @param [in]     b       B成分です.
    //! @param [in]     a       A成分です.
    //------------------------------------------------------------------------------------------
    void SetColor( f32 r, f32 g, f32 b, f32 a );

    //------------------------------------------------------------------------------------------
    //! @brief      テキストカラーを設定します.
    //!
    //! @param [in]     color       設定するカラー.
    //------------------------------------------------------------------------------------------
    void SetColor( const asdx::Vector4& color );

    //------------------------------------------------------------------------------------------
    //! @brief      描画を開始します.
    //------------------------------------------------------------------------------------------
    void Begin( ID3D11DeviceContext* pDeviceContext );

    //------------------------------------------------------------------------------------------
    //! @brief      文字列を描画します.
    //!
    //! @param [in]     x       描画開始X座標.
    //! @param [in]     y       描画開始Y座標.
    //! @param [in]     text    描画するテキスト.
    //------------------------------------------------------------------------------------------
    void DrawString( const s32 x, const s32 y, const char* text );

    //------------------------------------------------------------------------------------------
    //! @brief      文字列を描画します.
    //!
    //! @param [in]     x           描画開始X座標.
    //! @param [in]     y           描画開始Y座標.
    //! @param [in]     layerDepth  レイヤーの深さ.
    //! @param [in]     text        描画するテキスト.
    //------------------------------------------------------------------------------------------
    void DrawString( const s32 x, const s32 y, const s32 layerDepth, const char* text );

    //------------------------------------------------------------------------------------------
    //! @brief      文字列を描画します.
    //!
    //! @param [in]     x           描画開始X座標.
    //! @param [in]     y           描画開始Y座標.
    //! @param [in]     format      書式指定子.
    //! @param [in]     ...         可変引数.
    //------------------------------------------------------------------------------------------
    void DrawStringArg( const s32 x, const s32 y, const char* format, ... );

    //------------------------------------------------------------------------------------------
    //! @brief      文字列を描画します.
    //!
    //! @param [in]     x           描画開始X座標.
    //! @param [in]     y           描画開始Y座標.
    //! @param [in]     layerDepth  レイヤーの深さ.
    //! @param [in]     format      書式指定子.
    //! @param [in]     ...         可変引数.
    //------------------------------------------------------------------------------------------
    void DrawStringArg( const s32 x, const s32 y, const s32 layerDepth, const char* format, ... );

    //------------------------------------------------------------------------------------------
    //! @brief      描画終了処理です.
    //!
    //! @param [in]     pDeviceContext      デバイスコンテキスト.
    //------------------------------------------------------------------------------------------
    void End( ID3D11DeviceContext* pDeviceContext );

    //------------------------------------------------------------------------------------------
    //! @brief      フォントの横幅を取得します.
    //!
    //! @return     フォントの横幅を返却します.
    //------------------------------------------------------------------------------------------
    u32 GetFontWidth () const;

    //------------------------------------------------------------------------------------------
    //! @brief      フォントの縦幅を取得します.
    //!
    //! @return     フォントの縦幅を返却します.
    //------------------------------------------------------------------------------------------
    u32 GetFontHeight() const;

    //------------------------------------------------------------------------------------------
    //! @brief      フォント名を取得します.
    //!
    //! @return     フォント名を返却します.
    //------------------------------------------------------------------------------------------
    char*        GetFontName  () const;

    //------------------------------------------------------------------------------------------
    //! @brief      設定されているスクリーンサイズを取得します.
    //!
    //! @return     設定されているスクリーンサイズを返却します.
    //------------------------------------------------------------------------------------------
    asdx::Vector2 GetScreenSize() const;

    //------------------------------------------------------------------------------------------
    //! @brief      設定されているテキストカラーを取得します.
    //!
    //! @return     設定されているテキストカラーを返却します.
    //------------------------------------------------------------------------------------------
    asdx::Vector4 GetColor() const;
};

} // namespace asdx

#endif//__ASDX_FONT_H__
