﻿//---------------------------------------------------------------------------------------------
// File : asdxApp.h
// Desc : Application Module.
// Copyright(c) Project Asura. All right reserved.
//---------------------------------------------------------------------------------------------

#ifndef __ASDX_APP_H__
#define __ASDX_APP_H__

//---------------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------------
#include <d3d11.h>
#include <dxgi.h>
#include <Windows.h>
#include <DirectXMath.h>
#include <DirectXColors.h>
#include <asdxRenderTarget.h>
#include <asdxDepthStencilTarget.h>
#include <asdxMath.h>
#include <asdxTimer.h>


//---------------------------------------------------------------------------------------------
// Linker
//---------------------------------------------------------------------------------------------
#ifdef ASDX_AUTO_LINK
#pragma comment( lib, "d3d11.lib" )
#pragma comment( lib, "d3dcompiler.lib" )
#pragma comment( lib, "dxguid.lib" )
#pragma comment( lib, "winmm.lib" )
#pragma comment( lib, "comctl32.lib" )
#endif//ASDX_AUTO_LINK


//---------------------------------------------------------------------------------------------
// Macro
//---------------------------------------------------------------------------------------------
#ifndef ASDX_UNUSED_VAR
#define ASDX_UNUSED_VAR( x )        ((void)x)
#endif//ASDX_UNUSED_VAR


namespace asdx {

//////////////////////////////////////////////////////////////////////////////////////////////
// MouseEventParam structure
//////////////////////////////////////////////////////////////////////////////////////////////
struct MouseEventParam
{
    s32     X;                  //!< カーソルのX座標です.
    s32     Y;                  //!< カーソルのY座標です.
    s32     WheelDelta;         //!< マウスホイールの移動方向です.
    bool    IsLeftButtonDown;   //!< 左ボタンが押されたどうかを示すフラグです.
    bool    IsRightButtonDown;  //!< 右ボタンが押されたどうかを示すフラグです.
    bool    IsMiddleButtonDown; //!< 中ボタンが押されたかどうかを示すフラグです.
    bool    IsSideButton1Down;  //!< X1ボタンが押されたかどうかを示すフラグです.
    bool    IsSideButton2Down;  //!< X2ボタンが押されたかどうかを示すフラグです.

    //-----------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //-----------------------------------------------------------------------------------------
    MouseEventParam()
    : X                 ( 0 )
    , Y                 ( 0 )
    , WheelDelta        ( 0 )
    , IsLeftButtonDown  ( false )
    , IsRightButtonDown ( false)
    , IsMiddleButtonDown( false )
    , IsSideButton1Down ( false )
    , IsSideButton2Down ( false )
    { /* DO_NOTHING */ }
};


//////////////////////////////////////////////////////////////////////////////////////////////
// KeyEventParam structure
//////////////////////////////////////////////////////////////////////////////////////////////
struct KeyEventParam
{
    u32     KeyCode;        //!< キーコードです.
    bool    IsKeyDown;      //!< キーが押されたかどうかを示すフラグです.
    bool    IsAltDown;      //!< ALTキーが押されたかどうかを示すフラグです.

    //----------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------
    KeyEventParam()
    : KeyCode   ( 0 )
    , IsKeyDown ( false )
    , IsAltDown ( false )
    { /* DO_NOTHING */ }
};


//////////////////////////////////////////////////////////////////////////////////////////////
// ResizeEvent structure
//////////////////////////////////////////////////////////////////////////////////////////////
struct ResizeEventParam
{
    u32    Width;          //!< 画面の横幅です.
    u32    Height;         //!< 画面の縦幅です.
    f32    AspectRatio;    //!< 画面のアスペクト比です.

    //----------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------
    ResizeEventParam()
    : Width      ( 0 )
    , Height     ( 0 )
    , AspectRatio( 0.0f )
    { /* DO_NOTHING */ }
};


//////////////////////////////////////////////////////////////////////////////////////////////
// FrameEventParam struture
//////////////////////////////////////////////////////////////////////////////////////////////
struct FrameEventParam
{
    ID3D11DeviceContext*    pDeviceContext; //!< デバイスコンテキストです.
    f64                     Time;           //!< アプリケーション開始からの相対時間です.
    f64                     ElapsedTime;    //!< 前のフレームからの経過時間です.
    f32                     FPS;            //!< １秒当たりフレーム更新回数です.
    bool                    IsStopDraw;     //!< 描画停止フラグです.

    //----------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------
    FrameEventParam()
    : pDeviceContext( nullptr )
    , Time          ( 0 )
    , ElapsedTime   ( 0 )
    , FPS           ( 0.0f )
    , IsStopDraw    ( false )
    { /* DO_NOTHING */ }
};


//////////////////////////////////////////////////////////////////////////////////////////////
// Application class
//////////////////////////////////////////////////////////////////////////////////////////////
class Application
{
    //========================================================================================
    // list of friend classes and methods.
    //========================================================================================
    /* NOTHING */

public:
    //========================================================================================
    // public variables.
    //========================================================================================
    /* NOTHING */

    //========================================================================================
    // public methods.
    //========================================================================================

    //----------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------
    Application();

    //----------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     title       タイトル名.
    //! @param [in]     width       画面の横幅.
    //! @param [in]     height      画面の縦幅.
    //----------------------------------------------------------------------------------------
    Application( LPSTR title, u32 width, u32 height );

    //----------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //----------------------------------------------------------------------------------------
    virtual ~Application();

    //----------------------------------------------------------------------------------------
    //! @brief      アプリケーションを実行します.
    //----------------------------------------------------------------------------------------
    void    Run();

protected:
    //========================================================================================
    // protected variables
    //========================================================================================
    HINSTANCE                   m_hInst;                //!< インスタンスハンドルです.
    HWND                        m_hWnd;                 //!< ウィンドウハンドルです.
    D3D_DRIVER_TYPE             m_DriverType;           //!< ドライバータイプです.
    D3D_FEATURE_LEVEL           m_FeatureLevel;         //!< 機能レベルです.
    u32                         m_MultiSampleCount;     //!< マルチサンプリングのカウント数です.
    u32                         m_MultiSampleQuality;   //!< マルチサンプリングの品質値です.
    u32                         m_SwapChainCount;       //!< スワップチェインのバッファ数です.
    DXGI_FORMAT                 m_SwapChainFormat;      //!< スワップチェインのバッファフォーマットです.
    DXGI_FORMAT                 m_DepthStencilFormat;   //!< 深度ステンシルバッファのフォーマットです.
    ID3D11Device*               m_pDevice;              //!< デバイスです.
    ID3D11DeviceContext*        m_pDeviceContext;       //!< デバイスコンテキスト(イミディエイトコンテキスト)です.
    IDXGISwapChain*             m_pSwapChain;           //!< スワップチェインです.
    RenderTarget2D              m_RT;                   //!< 描画ターゲットです.
    DepthStencilTarget          m_DST;                  //!< 深度ステンシルバッファです.
    ID3D11RasterizerState*      m_pRS;                  //!< ラスタライザーステートです.
    ID3D11DepthStencilState*    m_pDSS;                 //!< 深度ステンシルステートです.
    ID3D11BlendState*           m_pBS;                  //!< ブレンドステートです.
    f32                         m_BlendFactor[ 4 ];     //!< ブレンドファクターです.
    u32                         m_SampleMask;           //!< サンプルマスクです.
    u32                         m_StencilRef;           //!< ステンシル参照です.
    asdx::Vector4               m_ClearColor;           //!< 背景のクリアカラーです.
    u32                         m_Width;                //!< 画面の横幅です.
    u32                         m_Height;               //!< 画面の縦幅です.
    f32                         m_AspectRatio;          //!< 画面のアスペクト比です.
    LPSTR                       m_Title;                //!< アプリケーションのタイトル名です.
    Timer                       m_Timer;                //!< タイマーです.

    //========================================================================================
    // protected methods.
    //========================================================================================
    
    //----------------------------------------------------------------------------------------
    //! @brief      初期化時に実行する処理です.
    //!
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual bool OnInit       ();

    //----------------------------------------------------------------------------------------
    //! @brief      終了時に実行する処理です.
    //!
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual void OnTerm       ();

    //----------------------------------------------------------------------------------------
    //! @brief      フレーム遷移時に実行する処理です.
    //!
    //! @param [in]     param       フレームインベントパラメータです.
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual void OnFrameMove  ( FrameEventParam& param );

    //----------------------------------------------------------------------------------------
    //! @brief      フレーム描画時に実行する処理です.
    //!
    //! @param [in]     param       フレームインベントパラメータです.
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual void OnFrameRender( FrameEventParam& param );

    //----------------------------------------------------------------------------------------
    //! @brief      リサイズ時に実行する処理です.
    //!
    //! @param [in]     param       リサイズイベントパラメータです.
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual void OnResize     ( const ResizeEventParam& param );

    //----------------------------------------------------------------------------------------
    //! @brief      キーイベント通知時に実行する処理です.
    //!
    //! @param [in]     param       キーイベントパラメータです.
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual void OnKey        ( const KeyEventParam&    param );

    //----------------------------------------------------------------------------------------
    //! @brief      マウスイベント通知時に実行する処理です.
    //!
    //! @param [in]     param       マウスイベントパラメータです.
    //! @note       派生クラスにて実装を行います.
    //----------------------------------------------------------------------------------------
    virtual void OnMouse      ( const MouseEventParam&  param );

    //----------------------------------------------------------------------------------------
    //! @brief      描画停止フラグを設定します.
    //!
    //! @param [in]     isStopRendering     描画を停止するかどうか.停止する場合はtrueを指定します.
    //----------------------------------------------------------------------------------------
    void SetStopRendering( bool isStopRendering );

    //----------------------------------------------------------------------------------------
    //! @brief      描画停止フラグを取得します.
    //!
    //! @retval true    描画処理を呼び出しません.
    //! @retval false   描画処理を呼び出します.
    //----------------------------------------------------------------------------------------
    bool IsStopRendering() const;

    //----------------------------------------------------------------------------------------
    //! @brief      フレームカウントを取得します.
    //!
    //! @return     フレームカウントを返却します.
    //----------------------------------------------------------------------------------------
    DWORD GetFrameCount() const;

    //----------------------------------------------------------------------------------------
    //! @brief      FPSを取得します.
    //!
    //! @return     FPSを返却します.
    //----------------------------------------------------------------------------------------
    f32 GetFPS() const;

    //----------------------------------------------------------------------------------------
    //! @brief      コマンドを実行して，画面に表示します.
    //!
    //! @param [in]     syncInterval        垂直同期の間隔です.
    //----------------------------------------------------------------------------------------
    void Present( u32 syncInterval );

private:
    //========================================================================================
    // private variables.
    //========================================================================================
    bool    m_IsStopRendering;      //!< 描画を停止するかどうかのフラグ. 停止する場合はtrueを指定.
    bool    m_IsStandbyMode;        //!< スタンバイモードかどうかを示すフラグです.
    DWORD   m_FrameCount;           //!< フレームカウントです.
    f32     m_FPS;                  //!< FPS(1秒あたりのフレーム描画回数)です.
    f64     m_LatestUpdateTime;     //!< 最後の更新時間です.

    //========================================================================================
    // private methods.
    //========================================================================================
    Application     ( const Application& );     // アクセス禁止.
    void operator = ( const Application& );     // アクセス禁止.

     //----------------------------------------------------------------------------------------
    //! @brief      アプリケーションの初期化処理です.
    //!
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //! @note       このメソッドは内部処理で，InitWnd(), InitD3D()を呼び出します.
    //----------------------------------------------------------------------------------------
    bool InitApp();

    //----------------------------------------------------------------------------------------
    //! @brief      アプリケーションの終了処理です.
    //!
    //! @note       このメソッドは内部処理で, TermD3D(), TermWnd()を呼び出します.
    //----------------------------------------------------------------------------------------
    void TermApp();

    //----------------------------------------------------------------------------------------
    //! @brief      ウィンドウの初期化処理です.
    //!
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //----------------------------------------------------------------------------------------
    bool InitWnd();

    //----------------------------------------------------------------------------------------
    //! @brief      ウィンドウの終了処理です.
    //----------------------------------------------------------------------------------------
    void TermWnd();

    //----------------------------------------------------------------------------------------
    //! @brief      Direct3Dの初期化処理です.
    //!
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //----------------------------------------------------------------------------------------
    bool InitD3D();

    //----------------------------------------------------------------------------------------
    //! @brief      Direct3Dの終了処理です.
    //----------------------------------------------------------------------------------------
    void TermD3D();

    //----------------------------------------------------------------------------------------
    //! @brief      メインループ処理です.
    //----------------------------------------------------------------------------------------
    void MainLoop();

    //----------------------------------------------------------------------------------------
    //! @brief      キーイベントを処理します.
    //!
    //! @param [in]     nChar       キーコード.
    //! @param [in]     isKeyDown   キーが押されているかどうか. キーが押されていればtrueを指定します.
    //! @param [in]     isAltDown   ALTキーが押されているかどうか. ALTキーが押されていればtrueを指定します.
    //! @note       このメソッドは内部処理で，OnKey()を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので,
    //!             アプリケーション側で呼び出しを行わないでください.
    //----------------------------------------------------------------------------------------
    void KeyEvent   ( u32 nChar, bool isKeyDown, bool isAltDown );

    //----------------------------------------------------------------------------------------
    //! @brief      リサイズイベントを処理します.
    //!
    //! @param [in]     width       画面の横幅.
    //! @param [in]     height      画面の縦幅.
    //! @note       このメソッドは内部処理で, OnResize()を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので,
    //!             アプリケーション側で呼び出しを行わないでください.
    //----------------------------------------------------------------------------------------
    void ResizeEvent( u32 width, u32 height );

    //----------------------------------------------------------------------------------------
    //! @brief      マウスイベントを処理します.
    //!
    //! @param [in]     x                   カーソルのX座標です.
    //! @param [in]     y                   カーソルのY座標です.
    //! @param [in]     wheelDelta          マウスホイールの移動方向です.
    //! @param [in]     isLeftButtonDown    左ボタンが押されたかどうか.押されていればtrueを指定します.
    //! @param [in]     isRightButtonDown   右ボタンが押されたかどうか.押されていればtrueを指定します.
    //! @param [in]     isMiddleButtonDown  中ボタンが押されたかどうか.押されていればtrueを指定します.
    //! @param [in]     isSideButton1Down   X1ボタンが押されたかどうか.押されていればtrueを指定します.
    //! @param [in]     isSideButton2Down   X2ボタンが押されたかどうか.押されていればtrueを指定します.
    //! @note       このメソッドは内部処理で, OnMouse()を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので,
    //!             アプリケーション側で呼び出しを行わないでください.
    //----------------------------------------------------------------------------------------
    void MouseEvent(
        s32     x,
        s32     y,
        s32     wheelDelta,
        bool    isLeftButtonDown,
        bool    isRightButtonDown,
        bool    isMiddleButtonDown,
        bool    isSideButton1Down,
        bool    isSideButton2Down );

    //----------------------------------------------------------------------------------------
    //! @brief      ウィンドウプロシージャです.
    //!
    //! @param [in]     hWnd        ウィンドウハンドル.
    //! @param [in]     uMsg        メッセージ.
    //! @param [in]     wp          メッセージの追加情報.
    //! @param [in]     lp          メッセージの追加情報.
    //----------------------------------------------------------------------------------------
    static LRESULT CALLBACK MsgProc( HWND hWnd, u32 uMsg, WPARAM wp, LPARAM lp );
};


} // namespace asdx

#endif//__ASDX_APP_H__
